﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Threading;

using MicroRWD.Common;
using MicroRWD.MFIC;

namespace MicroRWD.UI.MFIC
{
    public partial class ICODE_MemoryPanel : UserControl
    {
        private class ReadWorker
        {
            // TODO: Move C_LAST_BLOCK to a common constants location
            private const byte C_LAST_BLOCK = 0x1B;
            private byte startAddress = 0x00;
            private byte endAddress = C_LAST_BLOCK;

            private ICODE_MemoryPanel parent;
            private ReadCardProgress progress;

            public ReadWorker(ICODE_MemoryPanel _parent, ReadCardProgress _progress, byte _startAddress, byte _endAddress)
            {
                parent = _parent;
                progress = _progress;
                startAddress = ((_startAddress >= 0x00) && (_startAddress <= _endAddress) && (_startAddress <= (byte)C_LAST_BLOCK)) ? _startAddress : (byte)0x00;
                endAddress = ((_endAddress >= 0x00) && (_endAddress >= _startAddress) && (_endAddress <= C_LAST_BLOCK)) ? _endAddress : C_LAST_BLOCK;
            }

            public void DoWork()
            {
                try
                {
                    // Clear the listview
                    for (byte block = 0x00; block <= C_LAST_BLOCK; ++block)
                    {
                        // Clear data
                        parent.UpdateRow(block, "", "");
                    }

                    // Read card block by block
                    for (byte block = startAddress; (block <= endAddress) && !progress.Cancel; ++block)
                    {
                        // Delegate read to reader
                        byte[] reply = Program.Controller.Reader.CmdLabelBlockRead(block);

                        // Check for reply
                        if (reply.Length > 0)
                        {
                            // Check for data
                            if (reply.Length == 5)
                            {
                                // Extract data
                                byte[] data = new byte[4];
                                for (int i = 0; i < 4; ++i)
                                {
                                    data[i] = reply[i + 1];
                                }

                                // Copy data into list view
                                parent.UpdateRow(block, String.Join(string.Empty, Array.ConvertAll(data, b => b.ToString("X2") + ' ')), String.Join(string.Empty, Array.ConvertAll(data, b => ((b > 31) && (b < 127)) ? ((char)b).ToString() + ' ' : ". ")));
                            }
                            else
                            {
                                // Copy data into list view
                                parent.UpdateRow(block, "<NO DATA>", "");
                            }
                        }
                        else
                        {
                            // Clear data
                            parent.UpdateRow(block, "", "");
                        }

                        // Update progress bar
                        if (endAddress == 0)
                        {
                            parent.UpdateProgress(block * 100 / 1); // prevent divide by zero
                        }
                        else
                        {
                            parent.UpdateProgress(block * 100 / endAddress);
                        }
                    }

                    // Close progress dialog
                    parent.WorkerDone();
                }
                catch (Exception ex)
                {
                    // Error
                    Log.Error(ex.ToString());
                }
            }
        }

        // Create progress dialog
        private ReadCardProgress progress;
        private ReadCardMemory readCardMemory;

        public ICODE_MemoryPanel()
        {
            InitializeComponent();

            // Load Card Memory Definitions
            for (byte i = 0x00; i <= 0x1B; ++i)
            {
                cardListView.Items.Add(new ListViewItem(new string[] { i.ToString("X2"), "-- -- -- -- ", ". . . . ", "User Data" }));
            }

            // Select first item
            cardListView.Items[0].Selected = true;
        }

        #region Private Methods

        private void UpdateProgress(int _value)
        {
            this.UIThread(() => progress.Value = _value);
        }

        private void UpdateRow(int _row, string _hex, string _ascii)
        {
            this.UIThread(() => cardListView.Items[_row].SubItems[1].Text = _hex);
            this.UIThread(() => cardListView.Items[_row].SubItems[2].Text = _ascii);
        }

        private void WorkerDone()
        {
            this.UIThread(() => progress.Close());
        }

        private void writeToCard()
        {
            int selectedIndex = 0;
            if (cardListView.IsHandleCreated)
            {
                if (cardListView.SelectedIndices.Count > 0)
                {
                    selectedIndex = cardListView.SelectedIndices[0];
                }
                else
                {
                    selectedIndex = 0;
                }
            }
            else
            {
                selectedIndex = 0;
            }

            WriteToCardMemoryForm form = new WriteToCardMemoryForm(4, cardListView, selectedIndex, 3, Program.Controller.Reader);
            form.ShowDialog();
        }

        #region Event Handler

        private void ICODE_MemoryPanel_Load(object sender, EventArgs e)
        {
            // Register for status update events unless in design mode
            if (!DesignMode && (Program.Controller != null))
            {
                Program.Controller.Reader.StatusUpdateEvent += MyStatusChangedEventHandler;
            }
        }

        // Handler for status changed events
        private void MyStatusChangedEventHandler(object sender, StatusUpdateEventArgs args)
        {
            // Update Serial Number text with the latest value
            this.UIThread(() => serialNumberValueLabel.Text = Program.Controller.Reader.LastIcodeUidStr);
        }

        private void readCardButton_Click(object sender, EventArgs e)
        {
            // Create read card dialog
            readCardMemory = new ReadCardMemory(this);
            readCardMemory.ShowDialog();
        }

        public void readCard(byte _startAddress, byte _endAddress)
        {
            // Create progress dialog
            progress = new ReadCardProgress();

            // Create worker to do work asynchronously
            ReadWorker worker = new ReadWorker(this, progress, _startAddress, _endAddress);
            Thread t = new Thread(worker.DoWork);
            t.Start();
            progress.ShowDialog();
            t.Join();
        }

        private void writeToCardButton_Click(object sender, EventArgs e)
        {
            writeToCard();
        }

        private void cardListView_DoubleClick(object sender, EventArgs e)
        {
            writeToCard();
        }
                
        #endregion

        #endregion
    }
}
